#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmDiscardQuarantine;
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{

	[TestFixture]
	public class BOL_DiscardOrQuarantineBloodUnit : BaseTest 
	{
		#region Variables

		private const string _private_method_IsUnitAboRhInconsistent = "IsUnitAboRhInconsistent";
		private const string _private_method_IsUnitAntigenTypingInconsistent = "IsUnitAntigenTypingInconsistent";

		private BOL.BloodUnit _bloodUnit;
		private BOL.BloodUnit _bloodUnitTwo;
		private BOL.DiscardOrQuarantineBloodUnit _tstDQ;
			
		private Guid _bloodUnitGUID;
		private Guid _ccGuid;	//CannedCommentGuid

		#endregion

		#region SetUp

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				_ccGuid = DataCreator.GetRecordGuid("SELECT TOP 1 CannedCommentGuid FROM CannedComment WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");

				this._bloodUnit = DataCreator.CreateBloodUnit(false);
				this._bloodUnitTwo = DataCreator.CreateBloodUnit(false);

				this.RefreshData = false;
			}

			this._tstDQ = new gov.va.med.vbecs.BOL.DiscardOrQuarantineBloodUnit();
			this._tstDQ.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.Default);
			this._tstDQ.UserDateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			this._bloodUnitGUID = this._bloodUnit.BloodUnitGuid;
		}

		#endregion

		#region Test Properties

		[Test]
		public void CommentText_Pass()
		{
			string tstValue = DateTime.Now.ToString();

			this._tstDQ.CommentText = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.CommentText, "Value");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.CommentText), "BrokenRule.CommentText");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Details");
		}

		[Test]
		public void CommentText_Fail()
		{
			string tstValue = string.Empty;

			this._tstDQ.CommentText = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.CommentText, "Value");
			Assert.AreEqual(true, this._tstDQ.IsRuleBroken(BROKEN.CommentText), "BrokenRule.CommentText");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Details");
		}

		[Test]
		public void CommentText_Fail2()
		{
			string tstValue = "OTHER";

			this._tstDQ.CommentText = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.CommentText, "Value");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.CommentText), "BrokenRule.CommentText");
			Assert.AreEqual(true, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Details");
		}

		[Test]
		public void CommentGuid_Pass()
		{
			this._tstDQ.CommentGuid = this._ccGuid;

			Assert.AreEqual(this._ccGuid, this._tstDQ.CommentGuid, "Value");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CommentGuid_Fail()
		{
			// No BRs for this one
		}

		[Test]
		public void Details_Pass()
		{
			string tstValue = DateTime.Now.ToString();

			this._tstDQ.CommentText = "Other";
			this._tstDQ.Details = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.Details, "Value");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Details");
		}

		[Test]
		public void Details_Fail()
		{
			string tstValue = string.Empty;

			this._tstDQ.CommentText = "Other";
			this._tstDQ.Details = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.Details, "Value");
			Assert.AreEqual(true, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Details");

		}

		[Test]
		public void Action_Pass()
		{
			Common.Action tstValue = Common.Action.Discard;

			this._tstDQ.Action = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.Action);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Action_Fail()
		{
		}

		[Test]
		public void DiscardDisposition_Pass()
		{
			Common.DiscardDisposition tstValue = Common.DiscardDisposition.Credit;
			
			this._tstDQ.DiscardDisposition = tstValue;

			Assert.AreEqual(tstValue, _tstDQ.DiscardDisposition);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DiscardDisposition_Fail()
		{
		}

		[Test]
		public void BloodUnits_Pass()
		{
			System.Collections.ArrayList tstValue = new ArrayList();

			this._tstDQ.BloodUnits = tstValue;

			Assert.AreEqual(tstValue, _tstDQ.BloodUnits);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnits_Fail()
		{
		}

		[Test]
		public void UserDateTime_Pass()
		{
			DateTime tstValue = DateTime.Now.AddDays(-1);

			this._tstDQ.UserDateTime = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.UserDateTime, "Value");
			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.UserDateTime), "BrokenRule");
		}

		[Test]
		public void UserDateTime_Fail()
		{
			DateTime tstValue = DateTime.Now.AddDays(1);

			this._tstDQ.UserDateTime = tstValue;

			Assert.AreEqual(tstValue, this._tstDQ.UserDateTime, "Value");
			Assert.AreEqual(true, this._tstDQ.IsRuleBroken(BROKEN.UserDateTime), "BrokenRule");
		}

		#endregion

		#region Test Methods

		[Test]
		public void Ctor_Pass()
		{
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();

			Assert.IsNotNull(_tstDQ);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Fail()
		{
		}

		[Test]
		public void GetComboBoxEntries_Pass()
		{
			//QD is CannedComments Category Code
			DataTable dtTest = BOL.DiscardOrQuarantineBloodUnit.GetComboBoxEntries("QD");

			Assert.IsTrue(dtTest.Rows.Count > 0 );
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetComboBoxEntries_Fail()
		{
		}

		[Test]
		public void PopulateStatusComboBox_Pass()
		{
			DataTable dtTest = BOL.DiscardOrQuarantineBloodUnit.PopulateStatusComboBox();
			
			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PopulateStatusComboBox_Fail()
		{
		}

		[Test]
		public void LoadBloodUnitFields_Pass()
		{			
			this._tstDQ.LoadBloodUnitFields(this._bloodUnit);
			//If the call didn't raise an exception, it passed
			Assert.IsTrue(true);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void LoadBloodUnitFields_Fail()
		{
			this._tstDQ.LoadBloodUnitFields(null);
		}

		[Test]
		public void ChangeQuarantineStatus_Pass()
		{
			// TEST A.
			// 
			// Test with valid canned comment
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.CommentGuid = this._ccGuid;
			//
			Hashtable guidMap = DataCreator.CreateCrossmatchedBloodUnitTestData();
			Guid unitGuid = (Guid)guidMap["bloodUnitGuid"];
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Unit_quarantined, "UnitTest:ChangeQuarantineStatus_Pass", Common.LogonUser.LogonUserName);
			exRep.GenerateExceptionQuarantine(tmpBU);
			ArrayList exceptions  = new ArrayList();
			exceptions.Add(exRep);
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.DiscardUnit);
			ArrayList workloadProcessIds = new ArrayList(1);
			workloadProcessIds.Add(Common.WorkloadProcessID.DiscardUnit);
			bool test = tmpDQ.ChangeQuarantineStatus(exceptions, true, Common.UpdateFunction.UnitTests, workloadProcessIds);
			//
			Assert.IsTrue(test);
			//
			// Clear the patient test data
			DataKiller.DeleteCrossmatchedBloodUnitTestData(guidMap);
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());

			// ********************************************************************************************************************************
			// TEST B.
			//
			// Test with empty canned comment
			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.CommentGuid = Guid.Empty;
			//
			guidMap = DataCreator.CreateCrossmatchedBloodUnitTestData();
			unitGuid = (Guid)guidMap["bloodUnitGuid"];
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			exRep = new BOL.ExceptionReport(Common.ExceptionType.Unit_quarantined, "UnitTest:ChangeQuarantineStatus_Pass", Common.LogonUser.LogonUserName);
			exRep.GenerateExceptionQuarantine(tmpBU);
			exceptions  = new ArrayList();
			exceptions.Add(exRep);
			//
			
			test = tmpDQ.ChangeQuarantineStatus(exceptions, true, Common.UpdateFunction.UnitTests, workloadProcessIds);
			//
			Assert.IsTrue(test);
			//
			// Clear the patient test data
			DataKiller.DeleteCrossmatchedBloodUnitTestData(guidMap);
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void ChangeQuarantineStatus_Fail()
		{
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.QuarantineUnit);
			tmpDQ.BloodUnits.Add(new BOL.BloodUnit(Guid.NewGuid()));
			ArrayList workloadProcessIds = new ArrayList(1);
			workloadProcessIds.Add(Common.WorkloadProcessID.QuarantineUnit);
			//
			tmpDQ.ChangeQuarantineStatus(new ArrayList(), false, Common.UpdateFunction.UnitTests, workloadProcessIds);
		}

		[Test]
		public void ChangeDiscardStatus_Pass()
		{
			// Test with valid canned comment
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.CommentGuid = this._ccGuid;
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C") );
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.BloodUnitFinance.ReturnCreditAmount = 180;
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Unit_discarded, "UnitTest:ChangeDiscardStatus_Pass", Common.LogonUser.LogonUserName);
			exRep.GenerateExceptionQuarantine(tmpBU);
			ArrayList exceptions  = new ArrayList();
			exceptions.Add(exRep);
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.DiscardUnit);
			bool test = tmpDQ.ChangeDiscardStatus(exceptions, true, Common.UpdateFunction.UnitTests, Common.WorkloadProcessID.DiscardUnit);
			//
			Assert.IsTrue( test );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void ChangeDiscardStatus_Fail()
		{
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			tmpDQ.BloodUnits.Add(new BOL.BloodUnit(Guid.NewGuid()));
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.DiscardUnit);
			tmpDQ.ChangeDiscardStatus(new ArrayList(), false, Common.UpdateFunction.UnitTests,Common.WorkloadProcessID.DiscardUnit);
		}

		[Test]
		public void ValidateBatchSubmission_Discard_Pass()
		{
			#region Initialize

			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Discard;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_Discard_Pass";
			//
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime();
			tmpBU.DirectedPatient.PatientGuid = Guid.NewGuid();
			tmpBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpBU.BloodUnitStatus.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(1);
			tmpBU.BloodUnitStatus.CurrentStatusLastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(2);
			tmpBU.BloodUnitFinance.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(3);
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			DateTime[] invalidDate;
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( _tstDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.IsTrue ( validUnits.Count == 1 && failureMessages.Length == 0 && expiredUnits.Count == 0 && crossMatchedUnits.Count == 0 && expired.Length == 0 );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void ValidateBatchSubmission_Quarantine_Pass()
		{
			#region Initialize

			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Quarantine;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_Quarantine_Pass";
			//
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.DirectedPatient.PatientGuid = Guid.NewGuid();
			tmpBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			DateTime[] invalidDate;
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.IsTrue ( validUnits.Count == 1 && failureMessages.Length == 0 && expiredUnits.Count == 0 && crossMatchedUnits.Count == 0 && expired.Length == 0 );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());

			#region Initialize Test 2 : crossmatched unit 

			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Quarantine;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_Quarantine_Pass";
			//
			// create crossmatched blood unit
			Hashtable guidMap = DataCreator.CreateCrossmatchedBloodUnitTestData();
			DataCreator.InsertBloodUnitTest(Guid.NewGuid(), (Guid)guidMap["bloodUnitGuid"], (Guid)guidMap["orderedComponentGuid"], (Guid)guidMap["orderedUnitGuid"], 102, "C");
			unitGuid = (Guid)guidMap["bloodUnitGuid"];
			Guid patientGuid = (Guid)guidMap["patientGuid"];
			//
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.DirectedPatient.PatientGuid = patientGuid;
			tmpBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpBU.BloodUnitStatus.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddHours(-12);
			tmpBU.BloodUnitStatus.CurrentStatusLastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddHours(-11);
			tmpBU.BloodUnitMedia.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddHours(-10);
			tmpBU.BloodUnitFinance.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime().AddHours(-9);
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			failureMessages = new StringBuilder();
			warningMessages = new StringBuilder();
			expiredUnits = new ArrayList();
			crossMatchedUnits = new ArrayList();
			expired = new StringBuilder();

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.AreEqual(1, validUnits.Count, "1ValidUnits.Count");
			Assert.AreEqual(0, failureMessages.Length, "1FailureMessage.Length");
			Assert.AreEqual(0, expiredUnits.Count, "1ExpiredUnits.Count");
			Assert.AreEqual(0, crossMatchedUnits.Count, "1CrossmatchedUnits.Count");
			Assert.AreEqual(0, expired.Length, "1Expired.Length");

			Assert.IsTrue( FindMessage( warningMessages.ToString(), "The unit has been assigned/crossmatched, so if you quarantine it, then we will need to remove the assignment.\n" ) );
			//
			//
			// Clear the patient test data
			UnitTestUtil.RunSQL("DELETE FROM BLOODUNITTEST WHERE BLOODUNITGUID = '" + guidMap["bloodUnitGuid"].ToString() + "'");
			DataKiller.DeleteCrossmatchedBloodUnitTestData(guidMap);
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void ValidateBatchSubmission_RemoveFromQuarantine_Pass()
		{
			#region Initialize Test 1 : unit is not crossmatched or expired 

			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			//
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			DateTime[] invalidDate;
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.AreEqual(1, validUnits.Count, "1ValidUnits.Count");
			Assert.AreEqual(0, failureMessages.Length, "1FailureMessage.Length");
			Assert.AreEqual(0, expiredUnits.Count, "1ExpiredUnits.Count");
			Assert.AreEqual(0, crossMatchedUnits.Count, "1CrossmatchedUnits.Count");
			Assert.AreEqual(0, expired.Length, "1Expired.Length");
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());

			// ********************************************************************************************************
			#region Initialize Test 2 : crossmatched unit expected

			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			//
			// create crossmatched blood unit
			Hashtable guidMap = DataCreator.CreateCrossmatchedBloodUnitTestData();
			unitGuid = (Guid)guidMap["bloodUnitGuid"];
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			failureMessages = new StringBuilder();
			warningMessages = new StringBuilder();
			expiredUnits = new ArrayList();
			crossMatchedUnits = new ArrayList();
			expired = new StringBuilder();
			//
			// for this one, try sending an invalid user date < tmpBU.LastUpdateDate
			tmpDQ.UserDateTime = tmpBU.LastUpdateDate.Equals( DateTime.MinValue )
				? BOL.VBECSDateTime.GetDivisionCurrentDateTime().Subtract( new TimeSpan( 777, 0, 0, 0 ) ) 
				: tmpBU.LastUpdateDate.Subtract( new TimeSpan( 7, 0, 0, 0 ) );

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.AreEqual(1, validUnits.Count, "2ValidUnits.Count");
			Assert.AreEqual(0, failureMessages.Length, "2FailureMessage.Length");
			Assert.AreEqual(0, expiredUnits.Count, "2ExpiredUnits.Count");
			Assert.AreEqual(1, crossMatchedUnits.Count, "2CrossmatchedUnits.Count");
			Assert.AreEqual(0, expired.Length, "2Expired.Length");
			//
			// Clear the patient test data
			DataKiller.DeleteCrossmatchedBloodUnitTestData(guidMap);
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
			
			// ********************************************************************************************************
			#region Initialize Test 3 : expired unit expected

			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			//
			unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.DirectedPatient.PatientGuid = Guid.NewGuid();
			tmpBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpBU.BloodUnitMedia.UnitExpirationDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime().Subtract(new TimeSpan(1, 0, 0, 0));
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			failureMessages = new StringBuilder();
			warningMessages = new StringBuilder();
			expiredUnits = new ArrayList();
			crossMatchedUnits = new ArrayList();
			expired = new StringBuilder();
			//
			// for this one, try sending an invalid user date < tmpBU.BloodUnitStatus.LastUpdateDate
			tmpDQ.UserDateTime = tmpBU.BloodUnitStatus.LastUpdateDate.Equals( DateTime.MinValue )
				? BOL.VBECSDateTime.GetDivisionCurrentDateTime().Subtract( new TimeSpan( 777, 0, 0, 0 ) ) 
				: tmpBU.BloodUnitStatus.LastUpdateDate.Subtract( new TimeSpan( 7, 0, 0, 0 ) );


			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			//
			Assert.AreEqual(1, validUnits.Count, "3ValidUnits.Count");
			Assert.AreEqual(0, failureMessages.Length, "3FailureMessage.Length");
			Assert.AreEqual(1, expiredUnits.Count, "3ExpiredUnits.Count");
			Assert.AreEqual(0, crossMatchedUnits.Count, "3CrossmatchedUnits.Count");
			Assert.AreEqual(true, expired.Length > 0, "3Expired.Length");
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());

			// ********************************************************************************************************
			#region Initialize Test 4 : expired & crossmatched unit expected

			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			//
			// create crossmatched blood unit
			guidMap = DataCreator.CreateCrossmatchedBloodUnitTestData();
			unitGuid = (Guid)guidMap["bloodUnitGuid"];
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpBU.BloodUnitMedia.UnitExpirationDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime().Subtract(new TimeSpan(1, 0, 0, 0));
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			failureMessages = new StringBuilder();
			warningMessages = new StringBuilder();
			expiredUnits = new ArrayList();
			crossMatchedUnits = new ArrayList();
			expired = new StringBuilder();
			//
			// for this one, try sending an invalid user date < tmpBU.BloodUnitMedia.LastUpdateDate
			tmpDQ.UserDateTime = tmpBU.BloodUnitMedia.LastUpdateDate.Equals( DateTime.MinValue )
				? BOL.VBECSDateTime.GetDivisionCurrentDateTime().Subtract( new TimeSpan( 777, 0, 0, 0 ) ) 
				: tmpBU.BloodUnitMedia.LastUpdateDate.Subtract( new TimeSpan( 7, 0, 0, 0 ) );

			#endregion
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			
			// note: valid count is zero because "crossmatched unit has expired"
			Assert.AreEqual(0, validUnits.Count, "4ValidUnits.Count");
			Assert.AreEqual(true, failureMessages.Length > 0, "4FailureMessage.Length");
			Assert.AreEqual(0, expiredUnits.Count, "4ExpiredUnits.Count");
			Assert.AreEqual(1, crossMatchedUnits.Count, "4CrossmatchedUnits.Count");
			Assert.AreEqual(0, expired.Length, "4Expired.Length");
			//
			// Clear the patient test data
			DataKiller.DeleteCrossmatchedBloodUnitTestData(guidMap);
			//
			// Clear blood unit test data
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void ValidateBatchSubmission_Discard_Fail()
		{
			#region Initialize

			DateTime[] invalidDate;
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Discard;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_Discard_Fail";
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion
			//
			// Cannot discard when in Transfused status
			DataTable dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				new BOL.UnitTests.BOL_PatientTransfusionReaction().SaveImplicatedBloodUnits_Pass();
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			Guid finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			BOL.BloodUnit finalBU = new BOL.BloodUnit(finalGuid);
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			string expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("discard", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Transfused.Message" );
			Assert.AreEqual(0,  validUnits.Count, "Transfused.Count");
			//
			// Cannot discard when in Transferred status
			tmpDQ.BloodUnits.Clear();
			dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit tmpXUnit = DataCreator.CreateBloodUnit(true);
				string invoiceNumber = new Random().Next(1111111,9999999).ToString();
				BOL.LocalSupplier tmpLS = DataCreator.CreateLocalSupplier();
				BOL.OutgoingShipmentCollection tmpOSC = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice(invoiceNumber, tmpLS.LocalSupplierGuid, DateTime.Now, Common.ShipmentType.ReturnToSupplier, Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests);
				BOL.OutgoingShipmentCollection.LineItem lineItem = BOL.OutgoingShipmentCollection.LineItem.GetLineItemForUpdate(tmpXUnit.BloodUnitGuid);
				tmpOSC.AddUnitToInvoice(lineItem);
				tmpOSC.SaveChangesToInvoiceItems(new ArrayList(), Common.UpdateFunction.UnitTests);			//unit is not X, confirmation pending 
				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogout);
				tmpOSC.ConfirmShipment(DateTime.Now, Common.UpdateFunction.UnitTests, Common.WorkloadProcessID.UnitLogout);	//This puts the unit in X
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			finalBU = new BOL.BloodUnit(finalGuid);
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("discard", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Transferred.Message" );
			Assert.AreEqual(0, validUnits.Count,  "Transferred.Count");
			//
			// Cannot discard when in Discarded status
			tmpDQ.BloodUnits.Clear();
			dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'D' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit discardUnit = DataCreator.CreateBloodUnit(true);
				Guid commentGuid = DataCreator.GetRandomRecordGuid(TABLES.CannedComment.TableName,"");
				UnitTestUtil.RunSQL("UPDATE BloodUnitStatus SET UnitStatusComments = 'Test2373', UnitStatusDate = GetUtcDate(), DiscardCannedCommentGuid = '" + commentGuid.ToString() + "' WHERE BloodUnitGuid = '" + discardUnit.BloodUnitGuid.ToString() + "'");
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'D' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'D' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			finalBU = new BOL.BloodUnit(finalGuid);
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("discard", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Discarded.Message" );
			Assert.AreEqual(0, validUnits.Count, "Discarded.Count");
		}

		[Test]
		public void ValidateBatchSubmission_Quarantine_Fail()
		{
			#region Initialize

			DateTime[] invalidDate;
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Quarantine;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_Quarantine_Fail";
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion

			tmpDQ.BloodUnits.Clear();
			DataTable dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit tmpUnit = DataCreator.CreateRBCBloodUnit(true);
				DataCreator.CreateAboRhConfirmationTestsForUnit(tmpUnit.BloodUnitGuid, tmpUnit.BloodUnitMedia.AboRh);
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			Guid finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			BOL.BloodUnit finalBU = new BOL.BloodUnit(finalGuid);
			finalBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			string expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitAlreadyQuarantined(finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "AlreadyQuarantined.Message" );
			Assert.AreEqual(0, validUnits.Count, "AlreadyQuarantined.Count");
			//
			// Cannot quarantine when in Transfused status
			tmpDQ.BloodUnits.Clear();
			dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				new BOL.UnitTests.BOL_PatientTransfusionReaction().SaveImplicatedBloodUnits_Pass();
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			finalBU = new BOL.BloodUnit(finalGuid);
			finalBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("quarantine", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Transfused.Message" );
			Assert.AreEqual(0, validUnits.Count, "Transfused.Count");
			//
			// Cannot quarantine when in Transferred status
			tmpDQ.BloodUnits.Clear();
			dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit tmpXUnit = DataCreator.CreateBloodUnit(true);
				string invoiceNumber = new Random().Next(1111111,9999999).ToString();
				BOL.LocalSupplier tmpLS = DataCreator.CreateLocalSupplier();
				BOL.OutgoingShipmentCollection tmpOSC = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice(invoiceNumber, tmpLS.LocalSupplierGuid, DateTime.Now, Common.ShipmentType.ReturnToSupplier, Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests);
				BOL.OutgoingShipmentCollection.LineItem lineItem = BOL.OutgoingShipmentCollection.LineItem.GetLineItemForUpdate(tmpXUnit.BloodUnitGuid);
				tmpOSC.AddUnitToInvoice(lineItem);
				tmpOSC.SaveChangesToInvoiceItems(new ArrayList(), Common.UpdateFunction.UnitTests);			//unit is not X, confirmation pending 
				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogout);
				tmpOSC.ConfirmShipment(DateTime.Now, Common.UpdateFunction.UnitTests, Common.WorkloadProcessID.UnitLogout);	//This puts the unit in X
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'X' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");

			finalBU = new BOL.BloodUnit(finalGuid);
			finalBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("quarantine", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Transferred.Message" );
			Assert.AreEqual(0, validUnits.Count, "Transferred.Count");
			//
			// Cannot quarantine when in Discarded status
			tmpDQ.BloodUnits.Clear();
			dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'D' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit discardUnit = DataCreator.CreateBloodUnit(true);
				Guid commentGuid = DataCreator.GetRandomRecordGuid(TABLES.CannedComment.TableName,"");
				UnitTestUtil.RunSQL("UPDATE BloodUnitStatus SET UnitStatusComments = 'Test2373', UnitStatusDate = GetUtcDate(), DiscardCannedCommentGuid = '" + commentGuid.ToString() + "' WHERE BloodUnitGuid = '" + discardUnit.BloodUnitGuid.ToString() + "'");
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'D' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			finalBU = new BOL.BloodUnit(finalGuid);
			finalBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpDQ.BloodUnits.Add(finalBU);
			//
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus("quarantine", finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "Discarded.Message" );
			Assert.AreEqual(0, validUnits.Count, "Discarded.Count");
		}

		[Test]
		public void ValidateBatchSubmission_RemoveFromQuarantine_Fail()
		{
			#region Initialize

			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			tmpDQ.CommentText = "UnitTest:ValidateBatchSubmission_RemoveFromQuarantine_Fail";

			Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct WHERE ProductCode = '12000'");
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, bpGuid, true);

			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateAboRhTests = true;
			creatableTestType.CreateAntigenTests = true;
			creatableTestType.CreateBloodUnitAntigenRecord = true;
			creatableTestType.CreateInconsistentAboTest = true;
			creatableTestType.CreateInconsistentAntigenTest = true;
			creatableTestType.CreateInconsistentRhTest = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.DirectedPatient.PatientGuid = Guid.NewGuid();
			tmpBU.BloodUnitStatus.QuarantineIndicator = true;
			tmpDQ.BloodUnits.Add(tmpBU);
			//
			DateTime[] invalidDate;
			StringBuilder failureMessages = new StringBuilder();
			StringBuilder warningMessages = new StringBuilder();
			ArrayList expiredUnits = new ArrayList();
			ArrayList crossMatchedUnits = new ArrayList();
			StringBuilder expired = new StringBuilder();

			#endregion

			tmpDQ.BloodUnits.Clear();
			DataTable dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", false);
			if (dataCheck.Rows.Count == 0)
			{
				new BOL.UnitTests.BOL_PatientTransfusionReaction().SaveImplicatedBloodUnits_Pass();
				dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')", true);
			}
			Guid finalGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid from BloodUnitStatusCodeCurrent where UnitStatusCode = 'T' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			BOL.BloodUnit finalBU = new BOL.BloodUnit(finalGuid);
			finalBU.BloodUnitStatus.QuarantineIndicator = false;
			tmpDQ.BloodUnits.Add(finalBU);
			
			// Cannot remove when not already quarantined
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			ArrayList validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			string expectedMessage = Common.StrRes.SysErrMsg.UC027.UnitNotQuarantined(finalBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage(failureMessages.ToString(), expectedMessage), "NotQuarantined.Message" );
			Assert.AreEqual(0, validUnits.Count, "NotQuarantined.Count");
			//
			// Cannot remove when ABO\Rh is not consistent
			// Cannot quarantine when in Transfused status
			tmpDQ.BloodUnits.Clear();
			tmpDQ.BloodUnits.Add(tmpBU);

			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.InconsistentUnitAboRh(tmpBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage( failureMessages.ToString(), expectedMessage), "Transfused.Message" );
			Assert.AreEqual(0, validUnits.Count, "Transfused.Count");
			//
			// Cannot remove when antigen typing is not consistent
			invalidDate = new DateTime[tmpDQ.BloodUnits.Count];
			for (int i=0;i<tmpDQ.BloodUnits.Count;i++)
				invalidDate[i] = DateTime.MinValue;
			tmpDQ.UserDateTime = VBECSDateTime.GetDivisionCurrentDateTime().AddDays(-1);
			validUnits = tmpDQ.ValidateBatchSubmission( tmpDQ.Action, failureMessages, warningMessages, expiredUnits, crossMatchedUnits, expired, ref invalidDate );
			expectedMessage = Common.StrRes.SysErrMsg.UC027.InconsistentUnitAntigenTyping(tmpBU.EyeReadableUnitId).ResString;
			Assert.AreEqual(true, FindMessage( failureMessages.ToString(), expectedMessage), "AntigenTyping.Message" );
			Assert.AreEqual(0, validUnits.Count, "AntigenTyping.Count");
		}

		[Test]
		public void BuildValidString_Pass()
		{
			System.Collections.ArrayList alTest = new System.Collections.ArrayList();
			alTest.Add(this._bloodUnit);
			//
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			// Common.Action.Discard is the default
			string tstValue = tmpDQ.BuildValidString(alTest);
			Assert.IsTrue(FindMessage(tstValue, Common.StrRes.InfoMsg.Common.DiscardUnits().ResString));
			Assert.IsTrue(FindMessage(tstValue, this._bloodUnit.EyeReadableUnitId));
			//
			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.Quarantine;
			tstValue = tmpDQ.BuildValidString(alTest);
			Assert.IsTrue(FindMessage(tstValue, Common.StrRes.InfoMsg.Common.QuarantineUnits().ResString));
			Assert.IsTrue(FindMessage(tstValue, this._bloodUnit.EyeReadableUnitId));
			//
			tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();
			tmpDQ.Action = Common.Action.RemoveFromQuarantine;
			tstValue = tmpDQ.BuildValidString(alTest);
			Assert.IsTrue(FindMessage(tstValue, Common.StrRes.InfoMsg.Common.RemovefromQuarantineUnits().ResString));
			Assert.IsTrue(FindMessage(tstValue, this._bloodUnit.EyeReadableUnitId));
		}

		[Test]
		public void BuildValidString_Fail()
		{
			BOL.DiscardOrQuarantineBloodUnit tmpDQ = new BOL.DiscardOrQuarantineBloodUnit();

			string tstValue = tmpDQ.BuildValidString(new System.Collections.ArrayList());
			
			Assert.AreEqual(tstValue, Common.StrRes.InfoMsg.Common.NoValidUnits().ResString);
		}

		[Test]
		public void GetDiscardOrQuarantineUnits_Pass()
		{
			DataTable dt = BOL.DiscardOrQuarantineBloodUnit.GetDiscardOrQuarantineUnits();

			Assert.IsTrue(dt.Rows.Count >= 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetDiscardOrQuarantineUnits_Fail()
		{
		}

		[Test]
		public void SetInitialState_Pass()
		{
			this._tstDQ.CommentText = "Other";
			this._tstDQ.Details = string.Empty;

			Assert.AreEqual(true, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.Before");

			this._tstDQ.SetInitialState();

			Assert.AreEqual(false, this._tstDQ.IsRuleBroken(BROKEN.Details), "BrokenRule.After");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SetInitialState_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow drTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent").NewRow();

			DataRow dr2 = this._tstDQ.LoadDataRowFromThis(drTest);

			Assert.AreEqual(drTest, dr2);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}
	
		[Test]
		public void IsUnitAboRhInconsistent_No_Tests_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAboRhInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( !inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAboRhInconsistent_Consistent_Tests_Pass()
		{
			BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, Common.ComponentClass.RBC).BloodProductGuid, Common.ABO.A, Common.RH.Positive, this.TestDivisionCode, true);
            
			//Generic code in CreateAboRhTests assumes unit is A Pos
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateAboRhTests = true;
			CreateBloodUnitTests(tmpUnit.BloodUnitGuid , creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(tmpUnit.BloodUnitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAboRhInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( !inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAboRhInconsistent_Inconsistent_ABO_Test_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateAboRhTests = true;
			creatableTestType.CreateInconsistentAboTest = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAboRhInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAboRhInconsistent_Inconsistent_Rh_Test_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, Common.ABO.A, Common.RH.Negative, this.TestDivisionCode, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateAboRhTests = true;
			creatableTestType.CreateInconsistentRhTest = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAboRhInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsUnitAboRhInconsistent_Fail()
		{
		}

		[Test]
		public void IsUnitAntigenTypingInconsistent_No_Tests_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAntigenTypingInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( !inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAntigenTypingInconsistent_Blood_Unit_Antigen_Info_Only_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateBloodUnitAntigenRecord = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAntigenTypingInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( !inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAntigenTypingInconsistent_Consistent_Blood_Unit_Antigen_And_Antigen_Typing_Test_Data_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateBloodUnitAntigenRecord = true;
			creatableTestType.CreateAntigenTests = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAntigenTypingInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( !inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAntigenTypingInconsistent_Conflicting_Blood_Unit_Antigen_And_Antigen_Typing_Test_Data_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateBloodUnitAntigenRecord = true;
			creatableTestType.CreateAntigenTests = true;
			creatableTestType.CreateInconsistentAntigenTest = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAntigenTypingInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( inconsistent );
			//
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		public void IsUnitAntigenTypingInconsistent_Conflicting_Antigen_Typing_Tests_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._bloodUnit.BloodProductGuid, (this._bloodUnit.LabelTypeCode == "C"));
			//
			CreatableTestType creatableTestType = new CreatableTestType();
			creatableTestType.CreateAntigenTests = true;
			creatableTestType.CreateInconsistentAntigenTest = true;
			CreateBloodUnitTests(unitGuid, creatableTestType);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			//
			BOL.DiscardOrQuarantineBloodUnit _tstDQ = new BOL.DiscardOrQuarantineBloodUnit();
			//
			bool inconsistent = Convert.ToBoolean( _tstDQ.GetType().InvokeMember( _private_method_IsUnitAntigenTypingInconsistent, System.Reflection.BindingFlags.InvokeMethod | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic, null, _tstDQ, new object[]{ tmpBU.BloodUnitGuid } ) );
			//
			Assert.IsTrue( inconsistent );
			// 
			DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString());
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsUnitAntigenTypingInconsistent_Fail()
		{
		}
		
		#endregion

		#region Helper Classes / Methods

		/// <summary>
		/// Used to specify what types of Blood Unit Tests
		/// to create in CreateBloodUnit
		/// </summary>
		private sealed class CreatableTestType
		{
			#region Variables

			bool _createAboRhTests;
			bool _createInconsistentAboTest;
			bool _createInconsistentRhTest;
			//
			bool _createBloodUnitAntigenRecord;
			bool _createAntigenTests;
			bool _createInconsistentAntigenTest;

			#endregion
			//
			#region Properties

			/// <summary>
			/// CreateAboRhTests
			/// </summary>
			public bool CreateAboRhTests
			{
				get
				{
					return _createAboRhTests;
				}
				set
				{
					_createAboRhTests = value;
				}
			}

			/// <summary>
			/// CreateInconsistentAboTest
			/// </summary>
			public bool CreateInconsistentAboTest
			{
				get
				{
					return _createInconsistentAboTest;
				}
				set
				{
					_createInconsistentAboTest = value;
				}
			}

			/// <summary>
			/// CreateInconsistentRhTest
			/// </summary>
			public bool CreateInconsistentRhTest
			{
				get
				{
					return _createInconsistentRhTest;
				}
				set
				{
					_createInconsistentRhTest = value;
				}
			}
	
			/// <summary>
			/// CreateBloodUnitAntigenRecord
			/// </summary>
			public bool CreateBloodUnitAntigenRecord
			{
				get
				{
					return _createBloodUnitAntigenRecord;
				}
				set
				{
					_createBloodUnitAntigenRecord = value;
				}
			}

			/// <summary>
			/// CreateAntigenTests
			/// </summary>
			public bool CreateAntigenTests
			{
				get
				{
					return _createAntigenTests;
				}
				set
				{
					_createAntigenTests = value;
				}
			}

			/// <summary>
			/// CreateInconsistentAntigenTest
			/// </summary>
			public bool CreateInconsistentAntigenTest
			{
				get
				{
					return _createInconsistentAntigenTest;
				}
				set
				{
					_createInconsistentAntigenTest = value;
				}
			}

			#endregion
			//
			#region Constructor

			/// <summary>
			/// Constructor
			/// </summary>
			public CreatableTestType()
			{
				_createAboRhTests = false;
				_createInconsistentAboTest = false;
				_createInconsistentRhTest = false;
				//
				_createBloodUnitAntigenRecord = false;
				_createAntigenTests = false;
				_createInconsistentAntigenTest = false;
			}

			#endregion
		}

		/// <summary>
		/// CreateBloodUnitTests : From PatientUnitSelection (modified somewhat)
		/// </summary>
		private void CreateBloodUnitTests(Guid bloodUnitGuid, CreatableTestType creatableTestType)
		{
			#region Variables

			StringBuilder insert = new StringBuilder();
			string sql = string.Empty;
			
			#endregion
			//	
			#region Create ABO/Rh Tests

			if(creatableTestType.CreateAboRhTests)
			{
				// Insert 4 tests
				string testResultID = string.Empty;
				int bloodTestTypeID = 0;
				//
				for (int i = 0; i < 4; i++)
				{		
					if( i == 0 || ( i == 2 && !creatableTestType.CreateInconsistentAboTest ) )
					{
						bloodTestTypeID = (int) Common.TestType.ABOInterp;
						testResultID = Common.Utility.GetAboFromEnum(Common.ABO.A);
					}
					else if( i == 2 && creatableTestType.CreateInconsistentAboTest )
					{
						bloodTestTypeID = (int) Common.TestType.ABOInterp;
						testResultID = Common.Utility.GetAboFromEnum(Common.ABO.B);
					}
					else if( i == 1 || ( i == 3 && !creatableTestType.CreateInconsistentRhTest ) )
					{
						bloodTestTypeID = (int) Common.TestType.RhInterp;
						testResultID = Common.Utility.GetRhFromEnum(Common.RH.Positive);
					}
					else if( i == 3 && creatableTestType.CreateInconsistentRhTest )
					{
						bloodTestTypeID = (int) Common.TestType.RhInterp;
						testResultID = Common.Utility.GetRhFromEnum(Common.RH.Negative);
					}
					//
					insert.Append("INSERT INTO BloodUnitTest (BloodUnitTestGuid, BloodUnitGuid, ");
					insert.Append("BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, ");
					insert.Append("RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) ");
					insert.Append("VALUES ('");
					insert.Append(Guid.NewGuid());
					insert.Append("','");
					insert.Append(bloodUnitGuid);
					insert.Append("','");
					insert.Append(bloodTestTypeID);
					insert.Append("','");
					insert.Append(testResultID);
					insert.Append("','");
					insert.Append(Common.LogonUser.LogonUserName);
					insert.Append("','M','A','");
					insert.Append(Common.LogonUser.LogonUserDivisionCode);
					insert.Append("','");
					insert.Append(DateTime.Now.ToString());
					insert.Append("','");
					insert.Append(Common.LogonUser.LogonUserName);
					insert.Append("') ");
					UnitTestUtil.RunSQL(insert.ToString());
					insert.Remove(0, insert.Length);
				}
			}

			#endregion
			//
			#region Create Antigen Data

			// For our purposes, we will use 'c' Antigen (type 7, test id 224),
			// both for the BloodUnitAntigen record and the BloodUnitTest data

			if(creatableTestType.CreateBloodUnitAntigenRecord)
			{
				// Create BloodUnitAntigen record // used by UC01 and/or UC70
				//
				int antigenTypeId = 7;
				//
				insert.Append("INSERT INTO BloodUnitAntigen  ");
				insert.Append("(BloodUnitGuid, AntigenTypeId, Positive, Comments, DivisionCode, ");
				insert.Append("RecordStatusCode,LastUpdateDate, LastUpdateUser, LastUpdateFunctionId) ");
				insert.Append("VALUES ('");
				insert.Append(bloodUnitGuid.ToString()).Append("',");
				insert.Append(antigenTypeId).Append(",'");
				insert.Append("1', 'Unit Test:CreateBloodUnit','");
				insert.Append(Common.LogonUser.LogonUserDivisionCode).Append("','");
				insert.Append("A', '");
				insert.Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("','");
				insert.Append(Common.LogonUser.LogonUserName).Append("', 0)");
				UnitTestUtil.RunSQL(insert.ToString());
				insert.Remove(0, insert.Length);
			}

			if(creatableTestType.CreateAntigenTests)
			{
				int bloodTestTypeId = 224;
				string TestResultId = "P";
				Guid bloodUnitTestGuid = Guid.NewGuid();
				//
				insert.Append("INSERT INTO BloodUnitTest  ");
				insert.Append("(BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestDate, TestTechId, ");
				insert.Append("TestResultId, EntryTechId, EntryMethodCode, TestingMethodCode, TestComments, ");
				insert.Append("CorrectedResultIndicator, AntiseraWorklistTypingGuid, ");
				insert.Append("ConfirmationWorklistUnitGuid, OrderedTestGuid, OrderedComponentGuid, OrderedUnitGuid, ");
				insert.Append("InactivateDate, InactivateUser, PatientTransfusionReactionGuid, RackGuid, ");
				insert.Append("AutoInstrumentName, ReactivityPhaseCode, RecordStatusCode, DivisionCode, ");
				insert.Append("LastUpdateDate, LastUpdateUser, LastUpdateFunctionId )VALUES ('");
				insert.Append(bloodUnitTestGuid.ToString()).Append("','");
				insert.Append(bloodUnitGuid.ToString()).Append("',");
				insert.Append(bloodTestTypeId).Append(",'");
				insert.Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("','");
				insert.Append(Common.LogonUser.LogonUserName).Append("','");
				insert.Append(TestResultId).Append("','");
				insert.Append(Common.LogonUser.LogonUserName).Append("','");
				insert.Append("M','");
				insert.Append("G','");
				insert.Append("Unit Test:CreateBloodUnit',");
				insert.Append("NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, ");
				insert.Append("'A','");
				insert.Append(Common.LogonUser.LogonUserDivisionCode).Append("','");
				insert.Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("','");
				insert.Append(Common.LogonUser.LogonUserName).Append("', 0)");
				UnitTestUtil.RunSQL(insert.ToString());
				insert.Remove(0, insert.Length);
				//
				if(creatableTestType.CreateInconsistentAntigenTest)
				{
					TestResultId = "N";
					bloodUnitTestGuid = Guid.NewGuid();
					//
					insert.Append("INSERT INTO BloodUnitTest  ");
					insert.Append("(BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestDate, TestTechId, ");
					insert.Append("TestResultId, EntryTechId, EntryMethodCode, TestingMethodCode, TestComments, ");
					insert.Append("CorrectedResultIndicator, AntiseraWorklistTypingGuid, ");
					insert.Append("ConfirmationWorklistUnitGuid, OrderedTestGuid, OrderedComponentGuid, OrderedUnitGuid, ");
					insert.Append("InactivateDate, InactivateUser, PatientTransfusionReactionGuid, RackGuid, ");
					insert.Append("AutoInstrumentName, ReactivityPhaseCode, RecordStatusCode, DivisionCode, ");
					insert.Append("LastUpdateDate, LastUpdateUser, LastUpdateFunctionId )VALUES ('");
					insert.Append(bloodUnitTestGuid.ToString()).Append("','");
					insert.Append(bloodUnitGuid.ToString()).Append("',");
					insert.Append(bloodTestTypeId).Append(",'");
					insert.Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("','");
					insert.Append(Common.LogonUser.LogonUserName).Append("','");
					insert.Append(TestResultId).Append("','");
					insert.Append(Common.LogonUser.LogonUserName).Append("','");
					insert.Append("M','");
					insert.Append("G','");
					insert.Append("Unit Test:CreateBloodUnit',");
					insert.Append("NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, ");
					insert.Append("'A','");
					insert.Append(Common.LogonUser.LogonUserDivisionCode).Append("','");
					insert.Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("','");
					insert.Append(Common.LogonUser.LogonUserName).Append("', 0)");
					UnitTestUtil.RunSQL(insert.ToString());
					insert.Remove(0, insert.Length);
				}
			}

			#endregion
		}

		/// <summary>
		/// FindMessage
		/// </summary>
		private bool FindMessage(string message, string expectedErrorMessage)
		{
			string [] messages = message.Split(new char[]{ '\n' });
			//
			bool messageFound = false;
			//
			for (int idx = 0; idx < messages.Length; idx++)
			{
				if(messages[idx].TrimStart().TrimEnd().Equals(expectedErrorMessage.TrimStart().TrimEnd()))
				{
					messageFound = true;
					break;
				}
			}
			//
			return messageFound;
		}

		#endregion
	}
}
#endif




